function T_top = T_rise(nlayers,bidirectional,t,f_mod,fs,kr,kz,G,d,...
    Cv,Type,w0,radius,Power,alpha)

h = waitbar(0, 'Please Wait...');           % create waitbar
waitbar(1/4, h, sprintf('Initializing calculation'))
% function to do the temperature rise calculation 
% includes the integration function lgwt.m, written by Greg von Winckel - 02/25/2004

if min(abs(t))<20e-12
    min_t = 20e-12;
else
    min_t = min(min(abs(t)),100e-12);
end
Mcoeff = 1.5; % can scale number of terms to sum over
omega = 2*pi*f_mod;
omega_s = 2*pi*fs;
M = round(4*pi/(omega_s*min_t));
M = M*Mcoeff;
[k,wk]=lgwt(100,0,10/w0);         % Gauss nodes and weights for k

if strcmp(Type,'Pulsed')
    n = (-M):M;
elseif strcmp(Type,'CW')
    n = 0;
else
    error('Type must be CW or Pulsed')
end    

omega_vec = repmat(omega,[1,length(n)]) + n*omega_s; 
omega_mat = repmat(omega_vec,[length(k),1]); 
k_mat = repmat(k(:),[1,length(n)]);
wk_mat = repmat(wk(:),[1,length(n)]);
if ~(length(Cv(:)) == nlayers) || ~(length(kz(:)) == nlayers) || ~(length(kr(:)) == nlayers) || ...
        ~(length(d(:)) == nlayers) || ~(length(G(:)) == (nlayers-1))
    error('number of layers does not match property inputs')
end

if bidirectional
    % layer 1 = transparent substrate
    % layer 2 = metal substrate
    % layer 3 = substrate
    % heat deposited between layer 1 and 2
    q_glass = sqrt((kr(1).*k_mat.^2 + Cv(1).*1i.*omega_mat)./kz(1));
    q_metal = sqrt((kr(2).*k_mat.^2 + Cv(2).*1i.*omega_mat)./kz(2));
    q_liquid = sqrt((kr(3).*k_mat.^2 + Cv(3).*1i.*omega_mat)./kz(3));
    C_glass = -kz(1).*q_glass.*tanh(q_glass.*d(1));
    D_glass = 1;
    C_liquid = -kz(3).*q_liquid.*tanh(q_liquid*d(3));
    D_liquid = 1;
    A_metal = 1;
    A_liquid = 1;
    B_liquid = -1.*tanh(q_liquid*d(3))./(kz(3).*q_liquid);
    B_metal = -1.*tanh(q_metal.*d(2))./(kz(2).*q_metal);
    C_metal = -kz(2).*q_metal.*tanh(q_metal.*d(2));
    D_metal = 1;
    C1 = C_glass;
    D1 = C_glass*(-1/G(1)) + D_glass;
    C2 = C_liquid.*A_metal + C_liquid.*(-1/G(2)).*C_metal + D_liquid.*C_metal;
    D2 = C_liquid.*B_metal + C_liquid.*(-1/G(2)).*D_metal + D_liquid *D_metal;
    DC = -D1.*D2./(D1.*C2 + D2.*C1);
else
    for ll = 1:nlayers
        q(:,:,ll) = sqrt((kr(ll).*k_mat.^2 + Cv(ll).*1i.*omega_mat)./kz(ll));
        A(:,:,ll) = ones(length(k),length(n));
        B(:,:,ll) = -tanh(q(:,:,ll)*d(ll))./(kz(ll).*q(:,:,ll));
        C(:,:,ll) = -kz(ll).*q(:,:,ll).*tanh(q(:,:,ll)*d(ll));
        D(:,:,ll) = ones(length(k),length(n));
        if ll == 1
            Alay = A;
            Blay = B;
            Clay = C;
            Dlay = D;
        elseif ll > 1
            Alay(:,:,ll) = A(:,:,ll).*Alay(:,:,ll-1) - A(:,:,ll).*Clay(:,:,ll-1).*(1/G(ll-1)) + B(:,:,ll).*Clay(:,:,ll-1);
            Blay(:,:,ll) = A(:,:,ll).*Blay(:,:,ll-1) - A(:,:,ll).*Dlay(:,:,ll-1).*(1/G(ll-1)) + B(:,:,ll).*Dlay(:,:,ll-1);
            Clay(:,:,ll) = C(:,:,ll).*Alay(:,:,ll-1) - C(:,:,ll).*Clay(:,:,ll-1).*(1/G(ll-1)) + D(:,:,ll).*Clay(:,:,ll-1);
            Dlay(:,:,ll) = C(:,:,ll).*Blay(:,:,ll-1) - C(:,:,ll).*Dlay(:,:,ll-1).*(1/G(ll-1)) + D(:,:,ll).*Dlay(:,:,ll-1);
        else 
            error('something went wrong')
        end
    end
    DC = -Dlay(:,:,ll)./Clay(:,:,ll);
end

h = waitbar(2/4, h, sprintf('Calculating over radius'));
     
% to incorporate radial effects, expand tensor to 3rd order
DC_tens_r = repmat(DC,[1,1,length(radius)]);
k_tens_r = repmat(k_mat,[1,1,length(radius)]);
wk_tens_r = repmat(wk_mat,[1,1,length(radius)]);
radius2(1,1,:) = radius;
r_tens = repmat(radius2,length(k),length(n));
beam_shape = 1/(2*pi)*exp(-k_tens_r.^2.*w0^2/8); % gaussian, but can change if needed
T_top_FH = DC_tens_r.*beam_shape;
Q_top_FH = beam_shape;
f = DC_tens_r.*k_tens_r.*besselj(0,k_tens_r.*r_tens).*wk_tens_r.*beam_shape; 

%   f is the integrand to go from hankel space to radial space, 
%   dimension are [length(k),length(n),length(f_mod),length(radius)]

L0 = permute(sum(f,1),[2,3,1]);
%   L0 is the integegral of f over hankel space variable k
%   L0 has dimensions [length(n),length(radius)]

% T_top(r,w) = L0(r,w)*G(w)
% To get T_top(r,t), take inverse fourier transform -> simplifies to sum over n
% Before doing so, need to expand order once more to incorporate time

h = waitbar(3/4, h, sprintf('Calculating over time'));

omega_tens_t = repmat(omega_vec(:),[1,length(radius),length(t)]);
L0_tens = repmat(L0,[1,1,length(t)]);
t_tens = permute(repmat(t(:),[1,length(n),length(radius)]),[2,3,1]);
n_tens = repmat(n(:),[1,length(radius),length(t)]);
damper_tens = exp(-pi.*n_tens.^2./M^2); % optional damper
% damper_tens = 1;
T_top_components = Power*alpha*L0_tens.*exp((1i*omega_tens_t).*t_tens).*damper_tens;
% now dimensions are [length(n),length(radius),length(t)] 

% Fourier transform is equivalent to summing over n
T_top = permute(sum(T_top_components,1),[2,3,1]);
% T_top has dimensions [length(radius),length(t)]

h = waitbar(4/4, h, sprintf('Finished Calculation'));
close(h)

%---------------- Integration function ------------------------------------
function [x,w]=lgwt(N,a,b)
% lgwt.m
%
% This script is for computing definite integrals using Legendre-Gauss 
% Quadrature. Computes the Legendre-Gauss nodes and weights  on an interval
% [a,b] with truncation order N
%
% Suppose you have a continuous function f(x) which is defined on [a,b]
% which you can evaluate at any x in [a,b]. Simply evaluate it at all of
% the values contained in the x vector to obtain a vector f. Then compute
% the definite integral using sum(f.*w);
%
% Written by Greg von Winckel - 02/25/2004
N=N-1;
N1=N+1; N2=N+2;
xu=linspace(-1,1,N1)';
% Initial guess
y=cos((2*(0:N)'+1)*pi/(2*N+2))+(0.27/N1)*sin(pi*xu*N/N2);
% Legendre-Gauss Vandermonde Matrix
L=zeros(N1,N2);
% Derivative of LGVM
Lp=zeros(N1,N2);
% Compute the zeros of the N+1 Legendre Polynomial
% using the recursion relation and the Newton-Raphson method
y0=2;
% Iterate until new points are uniformly within epsilon of old points
while max(abs(y-y0))>eps
    L(:,1)=1;
    Lp(:,1)=0;
    L(:,2)=y;
    Lp(:,2)=1;
    for k=2:N1
        L(:,k+1)=( (2*k-1)*y.*L(:,k)-(k-1)*L(:,k-1) )/k;
    end
    Lp=(N2)*( L(:,N1)-y.*L(:,N2) )./(1-y.^2);   
    y0=y;
    y=y0-L(:,N2)./Lp; 
end
% Linear map from[-1,1] to [a,b]
x=(a*(1-y)+b*(1+y))/2;      
% Compute the weights
w=(b-a)./((1-y.^2).*Lp.^2)*(N2/N1)^2;    
end

end