
%% Calculate the surface temperature rise during TDTR or FDTR
% Jeff Braun, University of Viginia
% jb3gb@virginia.edu

% The T_rise.m function includes an integration function lgwt.m (by Greg von Winckel). Link:
% https://www.mathworks.com/matlabcentral/fileexchange/4540-legendre-gauss-quadrature-weights-and-nodes?focused=5053128&tab=function

% This code calculates the surface temperature rise (in time and radius) for a Gaussian 
% surface heating source (both pulsed and CW). You can specify any
% number of layers.

% In most cases (such as for estimating DC temperature rise in TDTR/FDTR), the
% surface temperature is a sufficient approximation, so for simplicity I
% left out the depth dependence of temperature.
% If you would like to calculate the full radial/depth temperature profile,
% I can send you the extended code to do so. Just send me an email.

% If you use any caluclations from this code for publication, please cite:
% "On the steady-state temperature rise during laser heating of multilayer 
% thin films in optical pump-probe techniques" by Jeffrey L. Braun,
% Chester J. Szwejkowski, Ashutosh Giri, and Patrick E. Hopkins (ASME
% Jounal of Heat Transfer)

%% Output is Surface_T_rise (probe averaged surface temperature rise)
% Surface_T_rise = probe averaged temperature rise vector with lenght(t)
% elements
% T_top = matrix of complex values, dimensions = [length(radius) length(t)]

%% Begin code
clear
close all
clc
tic 

%% User inputs:
% ------------------------------------------------------------------------------------------------
bidirectional = 0;                 % 0--> insulated surface, 1--> bidirectional heat flow
nlayers = 2;                       % number of layers, if bidirectional == 0 then it should be btw 1 & 4
fs = 80e6;                         % laser repetition rate in Hz (value doen't matter if 'CW' option is chosen)
t = 1e-12*linspace(100,12000,1);  % time (vector) in seconds
f_mod = 0;                      % modulation frequency (scalar) in Hz (f_mod = 0 is steady-state heating)
w0 = 15e-6;                        % pump radius in meters
w1 = 15e-6;                        % probe radius in meters             
alpha = 0.1;                       % absorbed fraction of the laser irradiation
Power = 10e-3;                     % laser power in Watts (absorbed power = alpha*Power)
radius = linspace(0,2*w0,301)';    % radius points of interest in meters
Type = 'CW';                       % Laser type ('Pulsed' or 'CW')

% Options for what information to plot
% ------------------------------------------------------------------------------------------------
plot_radial_temp = 1; % plots Temperature vs. radius for each time point (Fig. 1)
plot_pa_time = 1; % plots probe-averaged temperature vs. time (Fig. 2)

%------------------ Layers and Material Properties ------------------------
% Fill in the following vectors: number of terms = number of layers
% Cv = volumetric heat capacity [J/(m3 K)]
% kz = cross plane thermal conductivity [W/(m K)]
% kr = in plane thermal conductivity [W/(m K)]
% G = thermal boundary conductance [W/(m2 K)]
% d = film thickness [m], set to >>(sqrt(w0^2+w1^2)) if layer is
% semi-infinte, e.g. set to 1 

if bidirectional==0  % means insulated boundary condition
    % Vector index = layer #
    Cv = [2.42e6, 1.62e6];
    kz = [135, 1.4]; 
    kr = [135, 1.4]; 
    d = [80e-9, 1]; 
    G = [150e6]; % index# = TBC between layer # and (#+1) 
else  % means bidirectional
    % This version of the code only supports a 3 layer system,
    % where heat is deposited between layer 1 and layer 2
    % Heat deposited on the interface between first layer and second
    Cv = [1.62e6,2.42e6,4e6]; % order: glass, metal, liquid
    kz = [1.4, 135, .6];
    kr = [1.4, 135, .6];
    d = [1, 80e-9, 1];
    G = [100e6, 100e6];
end

%% Temperature Rise Calculation (no user input from here on):

T_top = T_rise(nlayers,bidirectional,t,f_mod,fs,kr,kz,G,d,Cv,Type,w0,radius,Power,alpha);

% Take real, imaginary, and magnitude:
T_top_real = real(T_top);
T_top_imag = imag(T_top);
T_top_mag = sqrt(T_top_real.^2+T_top_imag.^2);

% Numerically calculate probe averaged temperature rise (magnitude)
if length(radius) > 1
    for ll = 1:length(t)
        Surface_T_rise(ll) = 4/w1^2*trapz(radius(:),radius(:).*T_top(:,ll).*exp(-2*radius(:).^2/w1^2));
    end
else
    Surface_T_rise = T_top;
end

%% Plot Surface Temperatures
if plot_radial_temp
    figure(1)
    for ii = 1:length(t)
            clear T_toplot_real T_toplot_imag T_toplot_mag
            T_toplot_real(:) = T_top_real(:,ii);
            T_toplot_imag(:) = T_top_imag(:,ii);
            T_toplot_mag(:) = T_top_mag(:,ii);
            Mat_saved = [radius(:),T_toplot_real(:),T_toplot_imag(:),T_toplot_mag(:)];
            plot(radius*1e6,T_toplot_real,'bo')
            hold on
            plot(radius*1e6,T_toplot_imag,'ro')
            plot(radius*1e6,T_toplot_mag,'ko')
            legend_data1{3*ii-2} = ['Real, time =', num2str(t(ii)*1e9),'ns, freq = ', num2str(f_mod*1e-6), 'MHz'];
            legend_data1{3*ii-1} = ['Imaginary, time =', num2str(t(ii)*1e9),'ns, freq = ', num2str(f_mod*1e-6), 'MHz'];
            legend_data1{3*ii} = ['Magnitude, time =', num2str(t(ii)*1e9),'ns, freq = ', num2str(f_mod*1e-6), 'MHz'];
    end

    legend(legend_data1)
    xlabel('Radius (\mum)')
    ylabel('Surface temperature (K)')
    title('Surface temperature vs. radius')
end

if plot_pa_time
    % Plot the probe averaged surface temperature vs time
    figure(2)
    plot(t*1e9,real(Surface_T_rise(:)),'bo-')
    hold on
    plot(t*1e9,imag(Surface_T_rise(:)),'ro-')
    plot(t*1e9,abs(Surface_T_rise(:)),'ko-')
    xlabel('Time (ns)')
    ylabel('Surface temperature (K)')
    title('Probe-averaged surface temperature vs. time')
    legend('Real','Imaginary','Magnitude')
end

clearvars -except Surface_T_rise radius T_top t

toc
    
